<?php

namespace App\Jobs;

use App\Models\Invoice\Invoice;
use App\Services\ErrorHandler;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class GenerateInvoicePdf implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The invoice instance.
     *
     * @var \App\Models\Invoice\Invoice
     */
    protected $invoice;

    /**
     * The path where the PDF should be saved.
     *
     * @var string|null
     */
    protected $path;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The maximum number of unhandled exceptions to allow before failing.
     *
     * @var int
     */
    public $maxExceptions = 1;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(Invoice $invoice, ?string $path = null)
    {
        $this->invoice = $invoice;
        $this->path = $path;
        $this->onQueue('invoices');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        try {
            // Make sure invoice has an invoice number
            if (empty($this->invoice->invoice_number)) {
                $this->invoice->createInvoiceNumber();
            }

            // Generate and save PDF
            $pdfPath = $this->invoice->savePdf($this->path);

            return $pdfPath;
        } catch (\Exception $e) {
            $errorDetails = ErrorHandler::handleInvoiceError($e, 'PDF_GENERATION_JOB');

            app_log('Invoice PDF generation failed', 'error', $e, [
                'invoice_id' => $this->invoice->id,
                'error_details' => $errorDetails,
            ]);

            // Rethrow the exception to retry the job
            throw $e;
        }
    }
}
