<?php

namespace App\Jobs;

use App\Models\Invoice\Invoice;
use App\Services\Billing\BillingManager;
use App\Services\ErrorHandler;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ProcessInvoicePayment implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * The invoice instance.
     *
     * @var \App\Models\Invoice\Invoice
     */
    protected $invoice;

    /**
     * The number of times the job may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The maximum number of unhandled exceptions to allow before failing.
     *
     * @var int
     */
    public $maxExceptions = 1;

    /**
     * Indicate if the job should be marked as failed on timeout.
     *
     * @var bool
     */
    public $failOnTimeout = true;

    /**
     * The number of seconds the job can run before timing out.
     *
     * @var int
     */
    public $timeout = 120;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(Invoice $invoice)
    {
        $this->invoice = $invoice;
        $this->onQueue('payments');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle(BillingManager $billingManager)
    {
        try {
            // Skip if the invoice is already paid
            if ($this->invoice->isPaid()) {
                return;
            }

            // Skip if the invoice is free
            if ($this->invoice->isFree()) {
                $this->invoice->bypassPayment();

                return;
            }

            // Get the customer's default payment method
            $autoBillingData = $this->invoice->customer->getAutoBillingData();

            if (! $autoBillingData) {
                app_log('No auto-billing data found for customer', 'warning', null, [
                    'customer_id' => $this->invoice->customer->id,
                    'invoice_id' => $this->invoice->id,
                ]);

                return;
            }

            // Get the appropriate payment gateway
            $gateway = $billingManager->gateway($autoBillingData->type);

            if (! $gateway) {
                app_log('Payment gateway not found', 'warning', null, [
                    'gateway_type' => $autoBillingData->type,
                    'invoice_id' => $this->invoice->id,
                ]);

                return;
            }

            // Check if the gateway supports auto-billing
            if (! $gateway->supportsAutoBilling()) {
                app_log('Payment gateway does not support auto-billing', 'warning', null, [
                    'gateway_type' => $gateway->getType(),
                    'invoice_id' => $this->invoice->id,
                ]);

                return;
            }

            // Process the payment
            $gateway->autoCharge($this->invoice);

        } catch (\Exception $e) {
            $errorDetails = ErrorHandler::handlePaymentError($e, 'PAYMENT_JOB');

            app_log('Failed to process invoice payment', 'error', $e, [
                'invoice_id' => $this->invoice->id,
                'customer_id' => $this->invoice->customer->id,
            ]);
            // Rethrow the exception to retry the job
            throw $e;
        }
    }
}
