<?php

namespace App\Livewire\Admin\Page;

use App\Models\Page;
use App\Rules\PurifiedInput;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Livewire\Component;
use Stevebauman\Purify\Facades\Purify;

class ManagePages extends Component
{
    public Page $page;

    public $showPageModal = false;

    public $search = '';

    public $title = '';

    public $slug = '';

    public $parent_id;

    public $description = '';

    public $show_in_menu = false;

    public $status = true;

    public $order = 0;

    public $page_id = null;

    public $confirmingDeletion = false;

    protected $listeners = [
        'editPage' => 'editPage',
        'confirmDelete' => 'confirmDelete',
    ];

    protected function rules()
    {
        return [
            'page.title' => ['required', 'min:3', 'max:255', new PurifiedInput(t('sql_injection_error'))],
            'page.slug' => ['required', 'unique:pages,slug,'.($this->page->id ?? 'NULL'), new PurifiedInput(t('sql_injection_error'))],
            'page.description' => ['required'],
            'page.parent_id' => ['nullable'],
            'page.order' => ['nullable', 'numeric'],
            'page.show_in_menu' => ['nullable', 'boolean'],
            'page.status' => ['nullable', 'boolean'],
        ];
    }

    public function mount()
    {
        if (! checkPermission('admin.pages.view')) {
            $this->notify(['type' => 'danger', 'message' => t('access_denied_note')], true);

            return redirect(route('admin.dashboard'));
        }
        $this->resetForm();
        $this->page = new Page;
    }

    public function createPage()
    {
        $this->resetForm();
        $this->showPageModal = true;
    }

    private function resetForm()
    {
        $this->resetExcept('page');
        $this->resetValidation();
        $this->page = new Page;
    }

    public function save()
    {
        if (checkPermission(['admin.pages.create', 'admin.pages.edit'])) {
            $this->validate();
            try {
                if ($this->page->isDirty()) {
                    $this->page->save();
                    $this->showPageModal = false;
                    $this->notify([
                        'type' => 'success',
                        'message' => $this->page->wasRecentlyCreated
                            ? t('page_created_successfully')
                            : t('page_updated_successfully'),
                    ]);
                    Cache::forget('menu_items');

                    $this->dispatch('pg:eventRefresh-page-table');
                } else {
                    $this->showPageModal = false;
                }
            } catch (\Exception $e) {
                app_log('Page save failed: '.$e->getMessage(), 'error', $e, [
                    'page_id' => $this->page->id ?? null,
                ]);

                $this->notify(['type' => 'danger', 'message' => t('page_save_failed')]);
            }
        }
    }

    protected function sanitizeContent($content)
    {
        // First clean special Unicode characters
        $content = preg_replace('/[\x{200B}-\x{200D}\x{FEFF}]/u', '', $content);

        // Then apply HTML purifier
        $content = Purify::clean($content, [
            'HTML.Allowed' => 'a[href|title|target],p,br,b,i,u,strong,em,strike,blockquote,h1,h2,h3,h4,h5,h6,ol,ul,li,pre,code,div[class|style],span[class|style]',
        ]);

        // Remove empty paragraphs and multiple spaces
        $content = preg_replace('/<p[^>]*>(?:\s|&nbsp;)*<\/p>/', '', $content);
        $content = preg_replace('/\s+/', ' ', $content);

        return trim($content);
    }

    public function editPage($pageId)
    {
        $this->resetValidation();
        $page = Page::findOrFail($pageId);
        $this->page = $page;
        $this->resetValidation();
        Cache::forget('menu_items');

        $this->showPageModal = true;
    }

    public function confirmDelete($pageId)
    {
        $this->page_id = $pageId;
        $this->confirmingDeletion = true;
    }

    public function delete()
    {
        if (checkPermission('admin.pages.delete')) {
            try {
                Page::findOrFail($this->page_id)->delete();
                $this->confirmingDeletion = false;

                Cache::forget('menu_items');

                $this->notify(['type' => 'success', 'message' => t('page_deleted_successfully')]);
                $this->redirect(route('admin.pages'), navigate: true);
            } catch (\Exception $e) {
                app_log('Page deletion failed: '.$e->getMessage(), 'error', $e, [
                    'page_id' => $this->page_id ?? null,
                ]);

                $this->notify(['type' => 'danger', 'message' => t('page_delete_failed')]);
            }
        }
    }

    public function updatedPageTitle()
    {
        $this->slug = Str::slug($this->page->title);
    }

    public function getparentPagesProperty()
    {
        return Page::where('id', '!=', $this->page_id)
            ->orderBy('title')
            ->get();
    }

    public function refreshTable()
    {
        $this->dispatch('pg:eventRefresh-page-table');
    }

    public function render()
    {
        return view('livewire.admin.page.manage-pages');
    }
}
