<?php

namespace App\Livewire\Tenant\Contact;

use App\Models\Tenant\Contact;
use App\Models\Tenant\ContactNote;
use App\Services\FeatureService;
use App\Traits\WithTenantContext;
use Livewire\Component;

class ContactList extends Component
{
    use WithTenantContext;

    public Contact $contact;

    public ?int $contactId = null;

    public $confirmingDeletion = false;

    public $viewContactModal = false;

    public $contact_id = null;

    public array $selectedStatus = [];

    public $notes = [];

    public bool $isBulckDelete = false;

    protected $featureLimitChecker;

    public $tenant_id;

    public $tenant_subdomain;

    protected $listeners = [
        'editContact' => 'editContact',
        'confirmDelete' => 'confirmDelete',
        'viewContact' => 'viewContact',
    ];

    public function boot(FeatureService $featureLimitChecker)
    {
        $this->featureLimitChecker = $featureLimitChecker;

        $this->bootWithTenantContext();
    }

    public function mount()
    {
        if (! checkPermission(['tenant.contact.view', 'tenant.contact.view_own'])) {
            $this->notify(['type' => 'danger', 'message' => t('access_denied_note')], true);

            return redirect()->to(tenant_route('tenant.dashboard'));
        }

        $this->contact = new Contact;
        $this->tenant_id = tenant_id();
        $this->tenant_subdomain = tenant_subdomain_by_tenant_id($this->tenant_id);
    }

    public function createContact()
    {
        $this->redirect(tenant_route('tenant.contacts.save'));
    }

    public function viewContact($contactId)
    {
        // Initialize contact with correct table name
        if (! checkPermission('tenant.contact.view')) {
            $this->notify([
                'type' => 'danger',
                'message' => t('access_denied_note'),
            ]);

            return redirect()->to(tenant_route('tenant.dashboard'));
        }
        $this->contact = new Contact;
        $this->contact->setTable($this->tenant_subdomain.'_contacts');

        // Then load the contact
        $this->contact = Contact::fromTenant($this->tenant_subdomain)->findOrFail($contactId);
        $this->notes = ContactNote::fromTenant($this->tenant_subdomain)->where(['contact_id' => $contactId])->get();
        $this->viewContactModal = true;
    }

    public function importContact()
    {
        if (! checkPermission('tenant.contact.bulk_import')) {
            $this->notify([
                'type' => 'danger',
                'message' => t('access_denied_note'),
            ]);

            return redirect()->to(tenant_route('tenant.dashboard'));
        }

        return redirect()->to(tenant_route('tenant.contacts.imports'));
    }

    public function editContact($contactId)
    {
        $this->contact = Contact::fromTenant($this->tenant_subdomain)->findOrFail($contactId);
        $this->redirect(tenant_route('tenant.contacts.save', ['contactId' => $contactId]));
    }

    public function updatedConfirmingDeletion($value)
    {
        if (! $value) {
            $this->js('window.pgBulkActions.clearAll()');
        }
    }

    public function confirmDelete($contactId)
    {
        $this->contact_id = $contactId;

        $this->isBulckDelete = is_array($this->contact_id) && count($this->contact_id) !== 1 ? true : false;

        $this->confirmingDeletion = true;
    }

    public function delete()
    {
        if (checkPermission('tenant.contact.delete')) {
            if (is_array($this->contact_id) && count($this->contact_id) !== 0) {
                $selectedIds = $this->contact_id;
                // dispatch(function () use ($selectedIds) {
                Contact::fromTenant($this->tenant_subdomain)->whereIn('id', $selectedIds)
                    ->chunk(100, function ($contacts) {
                        foreach ($contacts as $contact) {
                            $contact->delete();
                        }
                    });
                // })->afterResponse();
                $this->contact_id = null;
                $this->js('window.pgBulkActions.clearAll()');
                $this->notify([
                    'type' => 'success',
                    'message' => t('contacts_delete_successfully'),
                ]);
            } else {

                $contact = Contact::fromTenant($this->tenant_subdomain)->findOrFail($this->contact_id);
                $this->contact_id = null;
                $contact->delete();

                $this->notify([
                    'type' => 'success',
                    'message' => t('contact_delete_success'),
                ]);
            }

            $this->confirmingDeletion = false;
            $this->dispatch('pg:eventRefresh-contact-table');
        }
    }

    public function getRemainingLimitProperty()
    {
        return $this->featureLimitChecker->getRemainingLimit('contacts', Contact::class);
    }

    public function getIsUnlimitedProperty()
    {
        return $this->remainingLimit === null;
    }

    public function getHasReachedLimitProperty()
    {
        return $this->featureLimitChecker->hasReachedLimit('contacts', Contact::class);
    }

    public function getTotalLimitProperty()
    {
        return $this->featureLimitChecker->getLimit('contacts');
    }

    public function refreshTable()
    {
        $this->dispatch('pg:eventRefresh-contact-table');
    }

    public function render()
    {
        return view('livewire.tenant.contact.contact-list');
    }
}
