<?php

namespace App\Livewire\Tenant\Tables;

use App\Models\Tenant\WhatsappTemplate;
use Illuminate\Database\Eloquent\Builder;
use PowerComponents\LivewirePowerGrid\Column;
use PowerComponents\LivewirePowerGrid\Facades\Filter;
use PowerComponents\LivewirePowerGrid\Facades\PowerGrid;
use PowerComponents\LivewirePowerGrid\PowerGridComponent;
use PowerComponents\LivewirePowerGrid\PowerGridFields;

final class WhatsppTemplateTable extends PowerGridComponent
{
    public string $tableName = 'whatspp-template-table-pygsun-table';

    public bool $showFilters = false;

    public bool $deferLoading = true;

    public string $loadingComponent = 'components.custom-loading';

    public function boot(): void
    {
        config(['livewire-powergrid.filter' => 'outside']);
    }

    public function setUp(): array
    {
        return [
            PowerGrid::header()
                ->withoutLoading()
                ->showToggleColumns()
                ->showSearchInput(),
            PowerGrid::footer()
                ->showPerPage()
                ->showRecordCount(),
        ];
    }

    public function datasource(): Builder
    {
        return WhatsappTemplate::selectRaw('*, ROW_NUMBER() OVER (ORDER BY id) as row_num')
            ->where('tenant_id', tenant_id());
    }

    public function relationSearch(): array
    {
        return [];
    }

    public function fields(): PowerGridFields
    {
        return PowerGrid::fields()
            ->add('row_num') 
            ->add(
                'status',
                fn($contact) => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ' .
                    ($contact->status === 'APPROVED'
                        ? 'bg-green-100 text-green-800 dark:text-green-400 dark:bg-green-900/20'
                        : 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300') .
                    '">' . ($contact->status ?? 'N/A') . '</span>'
            )
            ->add(
                'header_data_format',
                fn($contact) => $contact->header_data_format == null ? '-' : $contact->header_data_format
            );
    }

    public function columns(): array
    {
        return [
            Column::make(t('SR.NO'), 'row_num')
                ->sortable(),

            Column::make(t('template_name'), 'template_name')
                ->searchable()
                ->sortable(),

            Column::make(t('languages'), 'language')
                ->searchable()
                ->sortable(),

            Column::make(t('category'), 'category')
                ->searchable()
                ->sortable(),

            Column::make(t('template_type'), 'header_data_format')
                ->searchable()
                ->sortable(),

            Column::make(t('status'), 'status')
                ->searchable()
                ->sortable(),

            Column::make(t('body_data'), 'body_data')
                ->searchable()
                ->sortable()
                ->headerAttribute('text-wrap', 'white-space: normal;')
                ->bodyAttribute('text-wrap', 'white-space: normal; word-wrap: break-word;'),
        ];
    }

    public function filters(): array
    {
        return [
            Filter::inputText('template_name')->placeholder('Template Name'),

            $this->createSelectFilter('language'),
            $this->createSelectFilter('category'),
            $this->createFilterForHeaderDataFormat(),
            $this->createSelectFilter('status'),
        ];
    }

    // Regular select filter
    public function createSelectFilter(string $field)
    {
        return Filter::select($field, $field)
            ->dataSource(
                WhatsappTemplate::select($field)
                    ->distinct()
                    ->whereNotNull($field)
                    ->where($field, '!=', '')
                    ->pluck($field)
                    ->map(fn ($value) => ['id' => $value, 'name' => $value])
            )
            ->optionLabel('name')
            ->optionValue('id');
    }

    // Special filter for header_data_format to exclude nulls
    public function createFilterForHeaderDataFormat()
    {
        return Filter::select('header_data_format', 'header_data_format')
            ->dataSource(
                WhatsappTemplate::select('header_data_format')
                    ->distinct()
                    ->whereNotNull('header_data_format')
                    ->where('header_data_format', '!=', '')
                    ->pluck('header_data_format')
                    ->map(fn ($value) => ['id' => $value, 'name' => $value])
            )
            ->optionLabel('name')
            ->optionValue('id');
    }
}
