<?php

namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\Cache;

/**
 * Tenant Cache Service
 *
 * Provides optimized caching mechanisms for tenant data and settings in the multi-tenant
 * WhatsApp SaaS application. This service reduces database queries and improves performance
 * by implementing request-scoped caching strategies.
 *
 * @author WhatsApp SaaS Team
 *
 * @since 1.0.0
 *
 * Key Features:
 * - Request-scoped caching to prevent memory leaks in long-running processes
 * - Optimized database queries with selective column loading
 * - Cross-tenant setting retrieval without context switching overhead
 * - Automatic cache invalidation and cleanup mechanisms
 * @see \App\Models\Tenant For the underlying tenant model
 * @see \App\Facades\Tenant For facade access to tenant functionality
 *
 * @note This service uses request fingerprinting to ensure cache isolation
 *       between different requests and prevent data leakage.
 *
 * @warning Cache keys include request fingerprints, so they are unique per request.
 *          This prevents cross-request data pollution but may reduce cache hit rates.
 */
class TenantCacheService
{
    /**
     * Store tenant data in request-level cache
     *
     * @note Uses selective column loading to optimize memory usage and query performance.
     *       Only loads commonly needed columns: id, company_name, subdomain, domain, status, timestamps.
     *
     * @see self::getBySubdomain() For retrieving tenants by subdomain
     * @see self::forget() For cache invalidation
     */
    public static function remember(int $tenantId): ?Tenant
    {
        // Use request-scoped cache key
        $cacheKey = "tenant_{$tenantId}";

        return Cache::remember($cacheKey, 3600, function () use ($tenantId) {
            // Fetch tenant with only the columns typically needed
            return Tenant::find($tenantId);
        });
    }

    /**
     * Get tenant by subdomain from cache
     *
     * Retrieves tenant data using subdomain as the lookup key, with automatic
     * caching to improve performance for subdomain-based tenant resolution.
     *
     * @param  string  $subdomain  The tenant's subdomain identifier
     * @return Tenant|null The cached tenant instance or null if not found
     *
     * @example
     * ```php
     * $tenant = TenantCacheService::getBySubdomain('demo');
     * if ($tenant && $tenant->status === 'active') {
     *     // Process active tenant
     * }
     * ```
     *
     * @note This method is commonly used by middleware for tenant resolution
     *       in path-based multi-tenancy scenarios.
     *
     * @see \App\Http\Middleware\EnsureTenantExists For usage in tenant resolution
     * @see self::remember() For ID-based tenant retrieval
     */
    public static function getBySubdomain(string $subdomain): ?Tenant
    {
        // Use request-scoped cache key
        $cacheKey = "tenant_subdomain_{$subdomain}";

        return Cache::remember($cacheKey, 3600, function () use ($subdomain) {
            return Tenant::select([
                'id', 'company_name', 'subdomain', 'domain',
                'status', 'created_at', 'updated_at',
            ])
                ->where('subdomain', $subdomain)
                ->first();
        });
    }

    /**
     * Forget cached tenant
     *
     * Removes tenant data from the request-scoped cache, typically called
     * when tenant data is updated to ensure cache consistency.
     *
     * @param  int  $tenantId  The unique identifier of the tenant to remove from cache
     *
     * @example
     * ```php
     * // Update tenant and clear cache
     * $tenant->update(['status' => 'active']);
     * TenantCacheService::forget($tenant->id);
     * ```
     *
     * @note Only affects the current request's cache. Other concurrent
     *       requests maintain their own cache until expiration.
     *
     * @see self::remember() For caching tenant data
     */
    public static function forget(int $tenantId): void
    {
        $cacheKey = "tenant_{$tenantId}_".request()->fingerprint();
        Cache::forget($cacheKey);
    }
}
