<?php

namespace App\Traits;

use App\DTOs\WhatsAppMessage;
use App\Exceptions\WhatsAppException;
use App\Jobs\SendWhatsAppMessage;
use App\Models\Tenant\WhatsappTemplate;
use App\Models\Tenant\WmActivityLog;
use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\Label\Label;
use Endroid\QrCode\Logo\Logo;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\RoundBlockSizeMode;
use Endroid\QrCode\Writer\PngWriter;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Netflie\WhatsAppCloudApi\Message\ButtonReply\Button;
use Netflie\WhatsAppCloudApi\Message\ButtonReply\ButtonAction;
use Netflie\WhatsAppCloudApi\Message\CtaUrl\TitleHeader;
use Netflie\WhatsAppCloudApi\Message\Media\LinkID;
use Netflie\WhatsAppCloudApi\Message\Template\Component;
use Netflie\WhatsAppCloudApi\Response\ResponseException;
use Netflie\WhatsAppCloudApi\WhatsAppCloudApi;
use Throwable;

trait WhatsApp
{
    protected static string $facebookAPI = 'https://graph.facebook.com/';

    /**
     * Store WhatsApp settings to avoid multiple database calls
     */
    protected $whatsappSettings;

    protected $connectionSettings;

    /**
     * Load all WhatsApp settings in a single batch call
     */
    protected function loadWhatsAppSettings()
    {
        if (! isset($this->whatsappSettings)) {
            $this->whatsappSettings = get_batch_settings([
                'whatsapp.api_version',
                'whatsapp.wm_fb_app_id',
                'whatsapp.wm_fb_app_secret',
                'whatsapp.queue',
                'whatsapp.webhook_verify_token',
            ]);
        }

        return $this->whatsappSettings;
    }

    protected static array $extensionMap = [
        'image/jpeg' => 'jpg',
        'image/png' => 'png',
        'audio/mp3' => 'mp3',
        'video/mp4' => 'mp4',
        'audio/aac' => 'aac',
        'audio/amr' => 'amr',
        'audio/ogg' => 'ogg',
        'audio/mp4' => 'mp4',
        'text/plain' => 'txt',
        'application/pdf' => 'pdf',
        'application/vnd.ms-powerpoint' => 'ppt',
        'application/msword' => 'doc',
        'application/vnd.ms-excel' => 'xls',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
        'application/vnd.openxmlformats-officedocument.presentationml.presentation' => 'pptx',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'xlsx',
        'video/3gp' => '3gp',
        'image/webp' => 'webp',
    ];

    // Add this property to store tenant ID
    protected $wa_tenant_id = null;

    // Add this method to set tenant ID and enable method chaining
    public function setWaTenantId($tenant_id)
    {
        $this->wa_tenant_id = $tenant_id;

        return $this;
    }

    // Add this method to get current tenant ID with fallback
    protected function getWaTenantId()
    {
        return $this->wa_tenant_id ?? tenant_id();
    }

    protected static function getApiVersion(): string
    {
        // For static context, use regular get_setting
        $settings = get_batch_settings(['whatsapp.api_version']);

        return $settings['whatsapp.api_version'] ?? 'v21.0';
    }

    protected static function getBaseUrl(): string
    {
        return self::$facebookAPI.self::getApiVersion().'/';
    }

    protected function handleApiError(Throwable $e, string $operation, array $context = []): array
    {
        $tenant_id = $this->getWaTenantId();

        $errorContext = array_merge([
            'operation' => $operation,
            'account_id' => $this->getAccountID(),
            'phone_id' => $this->getPhoneID(),
            'tenant_id' => $tenant_id,
        ], $context);

        whatsapp_log("[WhatsApp {$operation} Error] ".$e->getMessage(), 'error', $errorContext, $e, $tenant_id);

        return [
            'status' => false,
            'message' => config('app.debug')
                ? $e->getMessage()
                : __('whatsapp.errors.'.$operation, ['default' => 'An error occurred during '.$operation]),
        ];
    }

    /**
     * Load core WhatsApp connection settings in a single batch call
     */
    protected function loadConnectionSettings()
    {
        if (! isset($this->connectionSettings)) {
            $this->connectionSettings = tenant_settings_by_group('whatsapp', $this->wa_tenant_id);
        }

        return $this->connectionSettings;
    }

    private function getToken(): ?string
    {
        $this->loadConnectionSettings();

        return $this->connectionSettings['wm_access_token'] ?? null;
    }

    private function getAccountID(): ?string
    {
        $this->loadConnectionSettings();

        return $this->connectionSettings['wm_business_account_id'] ?? null;
    }

    private function getPhoneID(): ?string
    {
        $this->loadConnectionSettings();

        return $this->connectionSettings['wm_default_phone_number_id'] ?? null;
    }

    private function getFBAppID(): ?string
    {
        $this->loadWhatsAppSettings();

        return $this->whatsappSettings['whatsapp.wm_fb_app_id'] ?? null;
    }

    private function getFBAppSecret(): ?string
    {
        $this->loadWhatsAppSettings();

        return $this->whatsappSettings['whatsapp.wm_fb_app_secret'] ?? null;
    }

    /**
     * Load WhatsApp Cloud API configuration
     *
     * @param  string|null  $fromNumber  Optional phone number to use as the sender
     * @return WhatsAppCloudApi Instance of the WhatsAppCloudApi class
     */
    public function loadConfig($fromNumber = null)
    {
        return new WhatsAppCloudApi([
            'from_phone_number_id' => (! empty($fromNumber)) ? $fromNumber : $this->getPhoneID(),
            'access_token' => $this->getToken(),
        ]);
    }

    public function getPhoneNumbers(): array
    {
        try {
            $response = Http::get(self::getBaseUrl()."{$this->getAccountID()}/phone_numbers", [
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json('data')];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'get_phone_numbers');
        }
    }

    public function loadTemplatesFromWhatsApp(): array
    {
        try {
            $accountId = $this->getAccountID();
            $accessToken = $this->getToken();
            $tenant_id = $this->getWaTenantId();

            $templates = [];
            $url = self::getBaseUrl()."{$accountId}/message_templates?limit=100&access_token={$accessToken}";

            // Fetch all templates using pagination
            do {
                $response = Http::get($url);

                if ($response->failed()) {
                    throw new WhatsAppException($response->json('error.message'));
                }

                $data = $response->json('data');
                if (! $data) {
                    throw new WhatsAppException('Message templates not found.');
                }

                $templates = array_merge($templates, $data);

                $url = $response->json('paging.next') ?? null;
            } while ($url);

            // Get existing template IDs from database to track what should be deleted
            $existingTemplateIds = WhatsappTemplate::where('tenant_id', $tenant_id)
                ->pluck('template_id')->toArray();
            $apiTemplateIds = [];

            foreach ($templates as $templateData) {
                $apiTemplateIds[] = $templateData['id'];
                $template = [
                    'template_name' => $templateData['name'],
                    'language' => $templateData['language'],
                    'status' => $templateData['status'],
                    'category' => $templateData['category'],
                    'id' => $templateData['id'],
                    'tenant_id' => $tenant_id,
                ];

                $components = [];
                $headerText = $bodyText = $footerText = $buttonsData = [];
                $headerParamsCount = $bodyParamsCount = $footerParamsCount = 0;

                foreach ($templateData['components'] as $component) {
                    if (($component['type'] ?? '') === 'HEADER') {
                        $components['TYPE'] = $component['format'] ?? null;
                        if (isset($component['text'])) {
                            $headerText = $component['text'];
                            $headerParamsCount = preg_match_all('/{{(.*?)}}/i', $headerText, $matches);
                            $components['HEADER'] = $headerText;
                        }
                    }
                    if (($component['type'] ?? '') === 'BODY' && isset($component['text'])) {
                        $bodyText = $component['text'];
                        $bodyParamsCount = preg_match_all('/{{(.*?)}}/i', $bodyText, $matches);
                        $components['BODY'] = $bodyText;
                    }
                    if (($component['type'] ?? '') === 'FOOTER' && isset($component['text'])) {
                        $footerText = $component['text'];
                        $footerParamsCount = preg_match_all('/{{(.*?)}}/i', $footerText, $matches);
                        $components['FOOTER'] = $footerText;
                    }
                    if (($component['type'] ?? '') === 'BUTTONS') {
                        $components['BUTTONS'] = isset($component['buttons']) ? json_encode($component['buttons']) : null;
                    }
                }

                $template['header_data_text'] = $components['HEADER'] ?? null;
                $template['header_data_format'] = $components['TYPE'] ?? null;
                $template['body_data'] = $components['BODY'] ?? null;
                $template['footer_data'] = $components['FOOTER'] ?? null;
                $template['buttons_data'] = $components['BUTTONS'] ?? null;
                $template['header_params_count'] = $headerParamsCount;
                $template['body_params_count'] = $bodyParamsCount;
                $template['footer_params_count'] = $footerParamsCount;
                unset($template['id']);

                WhatsappTemplate::updateOrCreate(
                    [
                        'template_id' => $templateData['id'],
                        'tenant_id' => $tenant_id,
                    ],
                    $template
                );
            }

            // Delete templates that exist in DB but not in API
            $templatesForDeletion = array_diff($existingTemplateIds, $apiTemplateIds);
            if (! empty($templatesForDeletion)) {
                $deletedCount = WhatsappTemplate::where('tenant_id', $tenant_id)
                    ->whereIn('template_id', $templatesForDeletion)
                    ->delete();

                whatsapp_log('Deleted templates during sync', 'info', [
                    'deleted_count' => $deletedCount,
                    'template_ids' => $templatesForDeletion,
                    'tenant_id' => $tenant_id,
                ], null, $tenant_id);
            }

            return [
                'status' => true,
                'data' => $templates,
                'synced' => [
                    'updated_or_created' => count($apiTemplateIds),
                    'deleted' => count($templatesForDeletion),
                ],
                'message' => t('templates_synced_successfully'),
            ];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'load_templates');
        }
    }

    public function subscribeWebhook()
    {
        $accessToken = $this->getToken();
        $accountId = $this->getAccountID();
        $tenant_id = $this->getWaTenantId();
        $url = self::$facebookAPI."/$accountId/subscribed_apps?access_token=".$accessToken;

        try {
            $response = Http::post($url);

            $data = $response->json();

            if (isset($data['error'])) {
                return [
                    'status' => false,
                    'message' => $data['error']['message'],
                ];
            }

            return [
                'status' => true,
                'data' => $data,
            ];
        } catch (\Throwable $th) {
            whatsapp_log('Failed to subscribe webhook: '.$th->getMessage(), 'error', [
                'url' => $url,
                'account_id' => $accountId,
                'tenant_id' => $tenant_id,
            ], $th, $tenant_id);

            return [
                'status' => false,
                'message' => 'Something went wrong: '.$th->getMessage(),
            ];
        }
    }

    public function queueMessage(WhatsAppMessage $message): array
    {
        try {
            $tenant_id = $this->getWaTenantId();

            // Add tenant ID to the message data
            $messageData = $message->toArray();
            $messageData['tenant_id'] = $tenant_id;

            // Create a new message instance with tenant ID
            $messageWithTenant = new WhatsAppMessage($messageData);

            // Load settings once if not already loaded
            $this->loadWhatsAppSettings();
            $queueSettings = json_decode($this->whatsappSettings['whatsapp.queue'] ?? '{"name":"default"}', true);

            dispatch(new SendWhatsAppMessage($messageWithTenant))
                ->onQueue($queueSettings['name']);

            return ['status' => true, 'message' => 'Message queued successfully'];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'queue_message', [
                'message' => $message->toArray(),
            ]);
        }
    }

    public function debugToken(): array
    {
        try {
            $accessToken = $this->getToken();
            $response = Http::get(self::getBaseUrl().'debug_token', [
                'input_token' => $accessToken,
                'access_token' => $accessToken,
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json('data')];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'debug_token');
        }
    }

    public function getProfile(): array
    {
        try {
            $response = Http::get(self::getBaseUrl().$this->getPhoneID().'/whatsapp_business_profile', [
                'fields' => 'profile_picture_url',
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json('data')];
        } catch (Throwable $e) {
            $data = $this->handleApiError($e, 'get_profile');

            return ['status' => false, 'data' => [], 'message' => $data['message'] ?? 'An error occurred while fetching the profile.'];
        }
    }

    public function getHealthStatus(): array
    {
        try {
            $response = Http::get(self::getBaseUrl().$this->getAccountID(), [
                'fields' => 'health_status',
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json()];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'health_status');
        }
    }

    public function getMessageLimit(): array
    {
        $startTime = strtotime(date('Y-m-d 00:00:00'));
        $endTime = strtotime(date('Y-m-d 23:59:59'));
        try {

            $response = Http::get(self::getBaseUrl().$this->getAccountID(), [
                'fields' => "id,name,analytics.start({$startTime}).end({$endTime}).granularity(DAY)",
                'access_token' => $this->getToken(),
            ]);

            if ($response->failed()) {
                throw new WhatsAppException($response->json('error.message'));
            }

            return ['status' => true, 'data' => $response->json()];
        } catch (Throwable $e) {
            $data = $this->handleApiError($e, 'health_status');

            return ['status' => false, 'data' => [], 'message' => $data['message']];
        }
    }

    public function generateUrlQR(string $url, ?string $logo = null): bool
    {
        try {
            $tenant_id = $this->getWaTenantId();
            $writer = new PngWriter;

            $qrCode = new QrCode(
                data: $url,
                encoding: new Encoding('UTF-8'),
                errorCorrectionLevel: ErrorCorrectionLevel::Low,
                size: 300,
                margin: 10,
                roundBlockSizeMode: RoundBlockSizeMode::Margin,
                foregroundColor: new Color(0, 0, 0),
                backgroundColor: new Color(255, 255, 255)
            );

            if ($logo) {
                $logo = new Logo(
                    path: public_path('img/whatsapp.png'),
                    resizeToWidth: 50,
                    punchoutBackground: true
                );
            }

            // Create generic label
            $label = new Label(
                text: '',
                textColor: new Color(255, 0, 0)
            );

            // Generate the QR code
            $result = $writer->write($qrCode, $logo, $label);

            create_storage_link();

            // Define the path to save the file
            $filePath = storage_path("app/public/tenant/{$tenant_id}/images/qrcode.png");

            // Ensure the directory exists
            if (! file_exists(dirname($filePath))) {
                mkdir(dirname($filePath), 0755, true);
            }

            // Save the QR code to the file
            $result->saveToFile($filePath);

            return true;
        } catch (Throwable $e) {
            $tenant_id = $this->getWaTenantId();
            whatsapp_log('Error generating QR code: '.$e->getMessage(), 'error', [
                'url' => $url,
                'logo' => $logo,
                'tenant_id' => $tenant_id,
            ], $e, $tenant_id);

            return false;
        }
    }

    public function connectWebhook($appId = null, $appSecret = null)
    {
        $appId = $appId ?? $this->getFBAppID();
        $appSecret = $appSecret ?? $this->getFBAppSecret();
        $tenant_id = $this->getWaTenantId();

        try {
            $url = self::$facebookAPI.$appId.'/subscriptions?access_token='.$appId.'|'.$appSecret;

            $response = Http::post($url, [
                'object' => 'whatsapp_business_account',
                'fields' => 'messages,message_template_quality_update,message_template_status_update,account_update',
                'callback_url' => route('whatsapp.webhook'),
                'verify_token' => $this->loadWhatsAppSettings()['whatsapp.webhook_verify_token'] ?? '',
            ]);

            $data = $response->json();

            if (isset($data['error'])) {
                return [
                    'status' => false,
                    'message' => $data['error']['message'],
                ];
            }

            return [
                'status' => true,
                'data' => $data,
            ];
        } catch (\Throwable $th) {
            whatsapp_log('Error connecting webhook: '.$th->getMessage(), 'error', [
                'tenant_id' => $tenant_id,
            ], $th, $tenant_id);

            return [
                'status' => false,
                'message' => 'Something went wrong: '.$th->getMessage(),
            ];
        }
    }

    public function disconnectWebhook()
    {
        $appId = $this->getFBAppID();
        $appSecret = $this->getFBAppSecret();
        $tenant_id = $this->getWaTenantId();

        $url = self::$facebookAPI.$appId.'/subscriptions?access_token='.$appId.'|'.$appSecret;

        try {
            $response = Http::delete($url, [], [
                'object' => 'whatsapp_business_account',
                'fields' => 'messages,message_template_quality_update,message_template_status_update,account_update',
            ]);

            $data = $response->json();

            if (isset($data['error'])) {
                return [
                    'status' => false,
                    'message' => $data['error']['message'],
                ];
            }

            return [
                'status' => true,
                'data' => $data,
            ];
        } catch (\Throwable $th) {
            whatsapp_log('Error disconnecting webhook: '.$th->getMessage(), 'error', [
                'tenant_id' => $tenant_id,
            ], $th, $tenant_id);

            return [
                'status' => false,
                'message' => 'Something went wrong: '.$th->getMessage(),
            ];
        }
    }

    public function sendTestMessages($number)
    {
        $tenant_id = $this->getWaTenantId();
        $whatsapp_cloud_api = $this->loadConfig();

        try {
            $result = $whatsapp_cloud_api->sendTemplate($number, 'hello_world', 'en_US');
            $status = true;
            $message = t('whatsapp_message_sent_successfully');
            $data = json_decode($result->body());
            $responseCode = $result->httpStatusCode();
        } catch (\Netflie\WhatsAppCloudApi\Response\ResponseException $th) {
            $status = false;
            $message = $th->responseData()['error']['message'] ?? $th->rawResponse() ?? json_decode($th->getMessage());
            $responseCode = $th->httpStatusCode();

            whatsapp_log('Error sending test message: '.$message, 'error', [
                'number' => $number,
                'response_code' => $responseCode,
                'tenant_id' => $tenant_id,
            ], $th, $tenant_id);
        }

        return ['status' => $status, 'message' => $message ?? ''];
    }

    public function checkServiceHealth(): array
    {
        try {
            $tenant_id = $this->getWaTenantId();

            // Load settings once if not already loaded
            $this->loadWhatsAppSettings();
            $queueSettings = json_decode($this->whatsappSettings['whatsapp.queue'] ?? '{"name":"default"}', true);

            $healthData = [
                'api_status' => $this->getHealthStatus(),
                'queue_size' => Queue::size($queueSettings['name']),
                'daily_api_calls' => Cache::get('whatsapp_api_calls_'.now()->format('Y-m-d')),
                'token_status' => $this->debugToken(),
                'profile_status' => $this->getProfile(),
                'tenant_id' => $tenant_id,
            ];

            whatsapp_log(
                'WhatsApp service health check',
                'info',
                $healthData,
                null,
                $tenant_id
            );

            return ['status' => true, 'data' => $healthData];
        } catch (Throwable $e) {
            return $this->handleApiError($e, 'health_check');
        }
    }

    protected function getExtensionForType(string $mimeType): ?string
    {
        return self::$extensionMap[$mimeType] ?? null;
    }

    /**
     * Send a template message using the WhatsApp Cloud API
     *
     * @param  string  $to  Recipient phone number
     * @param  array  $template_data  Data for the template message
     * @param  string  $type  Type of the message, default is 'campaign'
     * @param  string|null  $fromNumber  Optional sender phone number
     * @return array Response containing status, log data, and any response data or error message
     */
    public function sendTemplate($to, $template_data, $type = 'campaign', $fromNumber = null)
    {
        $tenant_id = $this->getWaTenantId();

        // CONVERSATION LIMIT CHECK FOR CAMPAIGNS
        $conversationTrackingNeeded = false;
        $identifierForTracking = null;

        if (($type === 'campaign' || $type == 'Initiate Chat') && ! empty($template_data['rel_id'])) {
            $featureService = app(\App\Services\FeatureService::class);
            $tenant_subdomain = tenant_subdomain_by_tenant_id($tenant_id);

            // Determine the conversation type and identifier
            $conversationType = $template_data['rel_type'] ?? 'guest';
            $identifierForCheck = $template_data['rel_id'];

            // Check if this would be a new conversation
            try {
                $hasActiveSession = $featureService->isConversationSessionActive(
                    $identifierForCheck,
                    $tenant_id,
                    $tenant_subdomain,
                    $conversationType
                );

                if (! $hasActiveSession) {
                    // This would be a new conversation
                    $conversationTrackingNeeded = true;
                    $identifierForTracking = $identifierForCheck;

                    // Check conversation limit
                    if ($featureService->checkConversationLimit($identifierForCheck, $tenant_id, $tenant_subdomain, $conversationType)) {
                        whatsapp_log('Campaign: Conversation limit reached - BLOCKING', 'warning', [
                            'to' => $to,
                            'identifier' => $identifierForCheck,
                            'type' => $conversationType,
                            'current_usage' => $featureService->getCurrentUsage('conversations'),
                            'current_limit' => $featureService->getLimit('conversations'),
                        ], null, $tenant_id);

                        $logdata = [
                            'status' => false,
                            'log_data' => [
                                'response_code' => 429,
                                'category' => $type,
                                'category_id' => $template_data['campaign_id'] ?? $template_data['template_bot_id'] ?? '',
                                'rel_type' => $conversationType,
                                'rel_id' => $identifierForCheck,
                                'response_data' => json_encode(['error' => 'Conversation limit reached']),
                                'tenant_id' => $tenant_id,
                                'category_params' => json_encode(['templateId' => $template_data['template_id'], 'message' => $message ?? '']),
                                'raw_data' => json_encode(['error' => 'Conversation limit reached']),
                                'phone_number_id' => $this->getPhoneID(),
                                'access_token' => $this->getToken(),
                                'business_account_id' => $this->getAccountID(),
                            ],
                            'data' => [],
                            'message' => t('conversation_limit_reached'),
                        ];
                        WmActivityLog::create($logdata['log_data']);

                        return $logdata;
                    }
                }
            } catch (\Exception $e) {
                whatsapp_log('Campaign: Error checking conversation limit', 'error', [
                    'to' => $to,
                    'identifier' => $identifierForCheck,
                    'error' => $e->getMessage(),
                ], $e, $tenant_id);
            }
        }

        // BUILD TEMPLATE COMPONENTS
        $rel_type = $template_data['rel_type'];
        $header_data = [];

        if ($template_data['header_data_format'] == 'TEXT') {
            $header_data = parseText($rel_type, 'header', $template_data, 'array');
        }
        $body_data = parseText($rel_type, 'body', $template_data, 'array');
        $buttons_data = parseText($rel_type, 'footer', $template_data, 'array');

        $component_header = $component_body = $component_buttons = [];
        $file_link = asset('storage/'.$template_data['filename']);

        $template_buttons_data = json_decode($template_data['buttons_data']);
        $is_flow = false;
        if (! empty($template_buttons_data)) {
            $button_types = array_column($template_buttons_data, 'type');
            $is_flow = in_array('FLOW', $button_types);
        }

        $component_header = $this->buildHeaderComponent($template_data, $file_link, $header_data);
        $component_body = $this->buildTextComponent($body_data);
        $component_buttons = $this->buildTextComponent($buttons_data);

        if ($is_flow) {
            $buttons = json_decode($template_data['buttons_data']);
            $flow_id = reset($buttons)->flow_id;
            $component_buttons[] = [
                'type' => 'button',
                'sub_type' => 'FLOW',
                'index' => 0,
                'parameters' => [
                    [
                        'type' => 'action',
                        'action' => [
                            'flow_token' => json_encode(['flow_id' => $flow_id, 'rel_data' => $template_data['flow_action_data'] ?? []]),
                        ],
                    ],
                ],
            ];
        }

        $whatsapp_cloud_api = $this->loadConfig($fromNumber);

        try {
            $components = new Component($component_header, $component_body, $component_buttons);
            $result = $whatsapp_cloud_api->sendTemplate($to, $template_data['template_name'], $template_data['language'], $components);
            $status = true;
            $data = json_decode($result->body());
            $responseCode = $result->httpStatusCode();
            $responseData = json_encode($result->decodedBody());
            $rawData = json_encode($result->request()->body());

            // TRACK CONVERSATION AFTER SUCCESSFUL SEND
            if ($status && $conversationTrackingNeeded && $identifierForTracking) {
                try {
                    $featureService = app(\App\Services\FeatureService::class);
                    $tenant_subdomain = tenant_subdomain_by_tenant_id($tenant_id);

                    $tracked = $featureService->trackNewConversation(
                        $identifierForTracking,
                        $tenant_id,
                        $tenant_subdomain,
                        $template_data['rel_type'] ?? 'guest'
                    );

                } catch (\Exception $e) {
                    whatsapp_log('Campaign: Failed to track conversation after send', 'error', [
                        'to' => $to,
                        'identifier' => $identifierForTracking,
                        'error' => $e->getMessage(),
                    ], $e, $tenant_id);
                }
            }
        } catch (\Netflie\WhatsAppCloudApi\Response\ResponseException $th) {
            $status = false;
            $message = $th->responseData()['error']['message'] ?? $th->rawResponse() ?? json_decode($th->getMessage());
            $responseCode = $th->httpStatusCode();
            $responseData = json_encode($message);
            $rawData = json_encode([]);

            whatsapp_log('Error sending template: '.$message, 'error', [
                'to' => $to,
                'template_name' => $template_data['template_name'],
                'language' => $template_data['language'],
                'response_code' => $responseCode,
                'response_data' => $responseData,
                'raw_data' => $rawData,
                'tenant_id' => $tenant_id,
            ], $th, $tenant_id);
        }

        $log_data = [
            'response_code' => $responseCode,
            'category' => $type,
            'category_id' => $template_data['campaign_id'] ?? $template_data['template_bot_id'],
            'rel_type' => $rel_type,
            'rel_id' => $template_data['rel_id'],
            'category_params' => json_encode(['templateId' => $template_data['template_id'], 'message' => $message ?? '']),
            'response_data' => $responseData,
            'raw_data' => $rawData,
            'phone_number_id' => $this->getPhoneID(),
            'access_token' => $this->getToken(),
            'business_account_id' => $this->getAccountID(),
            'tenant_id' => $tenant_id,
        ];

        // Create activity log
        WmActivityLog::create($log_data);

        return ['status' => $status, 'log_data' => $log_data, 'data' => $data ?? [], 'message' => $message->error->message ?? ''];
    }

    /**
     * Send a message using the WhatsApp Cloud API
     *
     * @param  string  $to  Recipient phone number
     * @param  array  $message_data  Data for the message
     * @param  string|null  $fromNumber  Optional sender phone number
     * @return array Response containing status, log data, and any response data or error message
     */
    public function sendMessage($to, $message_data, $fromNumber = null, $folder = 'bot_files')
    {
        $tenant_id = $this->getWaTenantId();
        $message_data = parseMessageText($message_data);
        $whatsapp_cloud_api = $this->loadConfig($fromNumber);

        try {
            $rows = [];
            if (! empty($message_data['button1_id'])) {
                $rows[] = new Button($message_data['button1_id'], $message_data['button1']);
            }
            if (! empty($message_data['button2_id'])) {
                $rows[] = new Button($message_data['button2_id'], $message_data['button2']);
            }
            if (! empty($message_data['button3_id'])) {
                $rows[] = new Button($message_data['button3_id'], $message_data['button3']);
            }
            if (! empty($rows)) {
                $action = new ButtonAction($rows);
                $result = $whatsapp_cloud_api->sendButton(
                    $to,
                    $message_data['reply_text'],
                    $action,
                    $message_data['bot_header'],
                    $message_data['bot_footer']
                );
            } elseif (! empty($message_data['button_name']) && ! empty($message_data['button_url']) && filter_var($message_data['button_url'], \FILTER_VALIDATE_URL)) {
                $header = new TitleHeader($message_data['bot_header']);

                $result = $whatsapp_cloud_api->sendCtaUrl(
                    $to,
                    $message_data['button_name'],
                    $message_data['button_url'],
                    $header,
                    $message_data['reply_text'],
                    $message_data['bot_footer'],
                );
            } else {
                $message = $message_data['bot_header']."\n".$message_data['reply_text']."\n".$message_data['bot_footer'];
                if (! empty($message_data['filename'])) {
                    $url = asset('storage/'.$message_data['filename']);
                    $link_id = new LinkID($url);
                    $fileExtensions = get_meta_allowed_extension();
                    $extension = strtolower(pathinfo($message_data['filename'], PATHINFO_EXTENSION));
                    $fileType = array_key_first(array_filter($fileExtensions, fn ($data) => in_array('.'.$extension, explode(', ', $data['extension']))));
                    if ($fileType == 'image') {
                        $result = $whatsapp_cloud_api->sendImage($to, $link_id, $message);
                    } elseif ($fileType == 'video') {
                        $result = $whatsapp_cloud_api->sendVideo($to, $link_id, $message);
                    } elseif ($fileType == 'document') {
                        $result = $whatsapp_cloud_api->sendDocument($to, $link_id, $message_data['filename'], $message);
                    }
                } else {
                    $result = $whatsapp_cloud_api->sendTextMessage($to, $message, true);
                }
            }

            $status = true;
            $data = json_decode($result->body());
            $responseCode = $result->httpStatusCode();
            $responseData = $data;
            $rawData = json_encode($result->request()->body());
        } catch (\Netflie\WhatsAppCloudApi\Response\ResponseException $th) {
            $status = false;
            $message = $th->responseData()['error']['message'] ?? $th->rawResponse() ?? $th->getMessage();
            $responseCode = $th->httpStatusCode();
            $responseData = $message;
            $rawData = json_encode([]);

            whatsapp_log('Error sending message: '.$message, 'error', [
                'to' => $to,
                'message_type' => $folder,
                'response_code' => $responseCode,
                'tenant_id' => $tenant_id,
            ], $th, $tenant_id);
        }

        $log_data = [
            'response_code' => $responseCode ?? 500,
            'category' => $folder == 'bot_files' ? 'message_bot' : '',
            'category_id' => $message_data['id'] ?? 0,
            'rel_type' => $message_data['rel_type'] ?? '',
            'rel_id' => $message_data['rel_id'] ?? '',
            'category_params' => json_encode(['message' => $message ?? '']),
            'response_data' => ! empty($responseData) ? json_encode($responseData) : '',
            'raw_data' => $rawData,
            'phone_number_id' => $this->getPhoneID(),
            'access_token' => $this->getToken(),
            'business_account_id' => $this->getAccountID(),
            'tenant_id' => $tenant_id,
        ];

        // Create activity log
        WmActivityLog::create($log_data);

        return ['status' => $status, 'log_data' => $log_data, 'data' => $data ?? [], 'message' => $message->error->message ?? ''];
    }

    /**
     * Send bulk campaign to WhatsApp recipients
     *
     * @param  string  $to  Recipient phone number
     * @param  array  $templateData  Template configuration
     * @param  array  $campaign  Campaign data
     * @param  string|null  $fromNumber  Sender phone number (optional)
     * @return array Response data
     */
    public function sendBulkCampaign($to, $templateData, $campaign, $fromNumber = null)
    {
        $tenant_id = $this->getWaTenantId();

        try {
            // Parse template data for header, body, and buttons
            $headerData = [];
            if ($templateData['header_data_format'] == 'TEXT') {
                $headerData = parseCsvText('header', $templateData, $campaign);
            }

            $bodyData = parseCsvText('body', $templateData, $campaign);
            $buttonsData = parseCsvText('footer', $templateData, $campaign);

            // Get file link if available
            $fileLink = ($templateData['filename']) ? asset('storage/'.$templateData['filelink']) : '';

            // Build components for WhatsApp message
            $componentHeader = $this->buildHeaderComponent($templateData, $fileLink, $headerData);
            $componentBody = $this->buildTextComponent($bodyData);
            $componentButtons = $this->buildTextComponent($buttonsData);

            // Load WhatsApp API configuration
            $whatsappCloudApi = $this->loadConfig($fromNumber);

            // Create components object and send template
            $components = new Component($componentHeader, $componentBody, $componentButtons);
            $result = $whatsappCloudApi->sendTemplate(
                $to,
                $templateData['template_name'],
                $templateData['language'],
                $components
            );

            return [
                'status' => true,
                'data' => json_decode($result->body(), true),
                'responseCode' => $result->httpStatusCode(),
                'message' => '',
                'phone' => $to,
                'tenant_id' => $tenant_id,
            ];
        } catch (ResponseException $e) {

            whatsapp_log('WhatsApp API Error: '.$e->getMessage(), 'error', [
                'phone' => $to,
                'template' => $templateData['template_name'],
                'response_code' => $e->httpStatusCode(),
                'response_data' => $e->responseData() ?? [],
                'tenant_id' => $tenant_id,
            ], $e, $tenant_id);

            return [
                'status' => false,
                'data' => [],
                'responseCode' => $e->httpStatusCode(),
                'message' => $e->responseData()['error']['message'] ?? $e->getMessage(),
                'phone' => $to,
                'tenant_id' => $tenant_id,
            ];
        } catch (\Exception $e) {

            whatsapp_log('WhatsApp Campaign Error: '.$e->getMessage(), 'error', [
                'phone' => $to,
                'template' => $templateData['template_name'] ?? 'unknown',
                'response_code' => 500,
                'tenant_id' => $tenant_id,
            ], $e, $tenant_id);

            return [
                'status' => false,
                'data' => [],
                'responseCode' => 500,
                'message' => $e->getMessage(),
                'phone' => $to,
                'tenant_id' => $tenant_id,
            ];
        }
    }

    /**
     * Retry sending a campaign message with exponential backoff
     *
     * @param  string  $to  Recipient phone number
     * @param  array  $templateData  Template configuration
     * @param  array  $campaign  Campaign data
     * @param  string|null  $fromNumber  Sender phone number (optional)
     * @param  int  $maxRetries  Maximum number of retry attempts
     * @return array Response data
     */
    public function sendWithRetry($to, $templateData, $campaign, $fromNumber = null, $maxRetries = 3)
    {
        $tenant_id = $this->getWaTenantId();
        $attempt = 0;
        $result = null;

        while ($attempt < $maxRetries) {
            $result = $this->sendBulkCampaign($to, $templateData, $campaign, $fromNumber);

            // If successful or not a retryable error, break the loop
            if ($result['status'] || ! $this->isRetryableError($result['responseCode'])) {
                break;
            }

            // Exponential backoff: wait longer between each retry
            $waitTime = pow(2, $attempt) * 1000000; // in microseconds (1s, 2s, 4s)
            usleep($waitTime);
            $attempt++;
        }

        return $result;
    }

    /**
     * Check if an error is retryable
     *
     * @param  int  $statusCode  HTTP status code
     * @return bool Whether the error is retryable
     */
    protected function isRetryableError($statusCode)
    {
        // Retry on rate limiting, server errors, and certain client errors
        return in_array($statusCode, [408, 429, 500, 502, 503, 504]);
    }

    /**
     * Handle batch processing for large campaigns
     *
     * @param  array  $recipients  List of recipients
     * @param  array  $templateData  Template configuration
     * @param  int  $batchSize  Batch size (default: 50)
     * @return array Results for each recipient
     */
    public function processBatchCampaign($recipients, $templateData, $batchSize = 50)
    {
        $tenant_id = $this->getWaTenantId();
        $results = [];
        $batches = array_chunk($recipients, $batchSize);

        foreach ($batches as $batch) {
            foreach ($batch as $recipient) {
                $to = $recipient['phone'];
                $result = $this->sendBulkCampaign($to, $templateData, $recipient);
                $results[] = $result;
            }

            // Add a small delay between batches to avoid rate limiting
            if (count($batches) > 1) {
                usleep(500000);
            }
        }

        return $results;
    }

    private function buildHeaderComponent($templateData, $fileLink, $headerData)
    {
        return match ($templateData['header_data_format']) {
            'IMAGE' => [['type' => 'image', 'image' => ['link' => $fileLink]]],
            'DOCUMENT' => [['type' => 'document', 'document' => ['link' => $fileLink, 'filename' => 'file_'.uniqid().'.'.pathinfo($templateData['filename'], PATHINFO_EXTENSION)]]],
            'VIDEO' => [['type' => 'video', 'video' => ['link' => $fileLink]]],
            default => collect($headerData)->map(fn ($header) => ['type' => 'text', 'text' => $header])->toArray(),
        };
    }

    private function buildTextComponent($data)
    {
        return collect($data)->map(fn ($text) => ['type' => 'text', 'text' => $text])->toArray();
    }

    /**
     * Retrieve a URL for a media file using its media ID
     *
     * @param  string  $media_id  Media ID to retrieve the URL for
     * @return string|null Filename of the saved media file or null on failure
     */
    public function retrieveUrl($media_id)
    {
        $tenant_id = $this->getWaTenantId();
        $url = self::$facebookAPI.$media_id;
        $accessToken = $this->getToken();

        $response = Http::withToken($accessToken)->get($url);

        if ($response->successful()) {
            $responseData = $response->json();

            if (isset($responseData['url'])) {
                $mediaUrl = $responseData['url'];
                $mediaData = Http::withToken($accessToken)->get($mediaUrl);

                if ($mediaData->successful()) {
                    $imageContent = $mediaData->body();
                    $contentType = $mediaData->header('Content-Type');

                    $extensionMap = self::$extensionMap;
                    $extension = $extensionMap[$contentType] ?? 'unknown';
                    $filename = 'media_'.uniqid().'.'.$extension;
                    $storagePath = 'whatsapp-attachments/'.$filename;

                    Storage::disk('public')->put($storagePath, $imageContent);

                    return $filename;
                }
            }
        }

        whatsapp_log('Failed to retrieve media URL', 'error', [
            'media_id' => $media_id,
            'tenant_id' => $tenant_id,
        ], null, $tenant_id);

        return null;
    }
}
